/*******************************************************************************
* Copyright (c) 2009 Luaj.org. All rights reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a copy
* of this software and associated documentation files (the "Software"), to deal
* in the Software without restriction, including without limitation the rights
* to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
* copies of the Software, and to permit persons to whom the Software is
* furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in
* all copies or substantial portions of the Software.
* 
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
* THE SOFTWARE.
******************************************************************************/
package org.luaj.vm2.lib.jme;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintStream;

import org.luaj.vm2.LuaTable;
import org.luaj.vm2.LuaUserdata;
import org.luaj.vm2.LuaValue;
import org.luaj.vm2.lib.BaseLib;
import org.luaj.vm2.lib.CoroutineLib;
import org.luaj.vm2.lib.LibFunction;
import org.luaj.vm2.lib.ResourceFinder;

/** 
 * Subclass of {@link BaseLib} and {@link LibFunction} which implements the lua basic library functions
 * and provides a directory based {@link ResourceFinder} as the {@link #FINDER}. 
 * <p>
 * Since JME has no file system by default, {@link BaseLib} implements 
 * {@link ResourceFinder} using {@link Class#getResource(String)}. 
 * The {@link JmeBaseLib} implements {@link FINDER} by scanning the current directory
 * first, then falling back to   {@link Class#getResource(String)} if that fails.
 * Otherwise, the behavior is the same as that of {@link BaseLib}.  
 * <p>  
 * Typically, this library is included as part of a call to 
 * {@link JmePlatform#standardGlobals()}
 * <p>
 * To instantiate and use it directly, 
 * link it into your globals table via {@link LuaValue#load(LuaValue)} using code such as:
 * <pre> {@code
 * LuaTable _G = new LuaTable();
 * LuaThread.setGlobals(_G);
 * _G.load(new JseBaseLib());
 * _G.get("print").call(LuaValue.valueOf("hello, world"));
 * } </pre>
 * Doing so will ensure the library is properly initialized 
 * and loaded into the globals table. 
 * <p>
 * This is a direct port of the corresponding library in C.
 * @see BaseLib
 * @see ResourceFinder
 * @see #FINDER
 * @see LibFunction
 * @see JsePlatform
 * @see JmePlatform
 * @see <a href="http://www.lua.org/manual/5.1/manual.html#5.1">http://www.lua.org/manual/5.1/manual.html#5.1</a>
 * @author <a href="http://www.arakhne.org">St&eacute;phane Galland</a>
 */

public class JmeBaseLib extends org.luaj.vm2.lib.BaseLib {

	/** Construct a JSE base library instance */
	public JmeBaseLib() {
		//
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	protected InputStream getStdIn(LuaValue v) {
		return stdin(v);
	}

	/**
	 * Replies the standard input stream associated to the specified value.
	 * 
	 * @param v
	 * @return the stream.
	 */
	static InputStream stdin(LuaValue v) {
		LuaValue s = v.getfenv().get(STDIN_KEY);
		if (s==null || s.isnil()) return System.in;
		return (InputStream) ((LuaUserdata)s).userdata();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected PrintStream getStdOut(LuaValue v) {
		return stdout(v);
	}
	
	/**
	 * Replies the standard output stream associated to the specified value.
	 * 
	 * @param v
	 * @return the stream.
	 */
	static PrintStream stdout(LuaValue v) {
		LuaValue s = v.getfenv().get(STDOUT_KEY);
		if (s==null || s.isnil()) return System.out;
		return (PrintStream) ((LuaUserdata)s).userdata();
	}

}
